// this file contains Linux library code for the Vaunix LSG synthesizer
//
// includes changes from AK to Linux attenuator library
/*	
	- Changed include <hid.h> to <linux/hid.h>, which is the correct
	location for the kernel headers in modern linux systems.

	- Added <stdbool.h> to includes, needed to make 'bool' types work.

	- Added <unistd.h> to includes in 'test.c' to make 'getlogin()' 
	work as expected. Otherwise, the 'test' program results in a 
	Segmentation Fault right at the beginning.

	- When running as an unprivileged user (i.e. not 'root'), the program
	suggests to try running with sudo (which is the preferred way to
	raise privilege temporarily), before suggesting to become root via
	'su'.	

-- Attila Kovacs <attila[AT]submm.caltech.edu>
*/


// RSD 4-15-2013 draft code for LSG synthesizers, derived from LMS library
// !untested!
//-----------------------------------------------------------------------------
#include <usb.h>
#include <linux/hid.h>	/* AK: Changed include for modern linux. */
#include <stdbool.h>	/* AK: Added include for 'bool' type */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include "LSGhid.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#define DEBUG_OUT 0  /* set this to 1 in order to see debugging output, 2 for a ton of output, or 3 for many tons */
#define FALSE 0
#define TRUE !FALSE

#define PACKET_CTRL_LEN 8
#define PACKET_INT_LEN 8
#define INTERFACE 0
#define ENDPOINT_INT_IN 0x82
#define TIMEOUT 500
#define LIBVER "0.80"

void *brick_handler_function (void *ptr);

/* ----------------------------------------------------------------- */
/* globals we'll be using at runtime */
char errmsg[32]; 			// For the status->string converter

bool bVerbose = FALSE; 		// True to generate debug oriented printf output

bool TestMode = TRUE; 		// if TestMode is true we fake it -- no HW access
							// TestMode defaults to FALSE for production builds

LSGPARAMS lsg [MAXDEVICES]; // an array of structures each of which holds the info for a given 
							// device. The DeviceID is the index into the array. Devices may come and go
							// so there is no guarantee that the active elements are contiguous

char lsgpath [MAXDEVICES*MAX_MODELNAME]; // space to store the device path for each of our devices

char sVNX1[32] = "LSG-402";
char sVNX2[32] = "LSG-152";
char sVNX3[32] = "LSG-251";
char sVNX4[32] = "LSG-602";
char sVNX5[32] = "LSG-222";
char sVNX6[32] = "LSG-121";
char sVNX7[32] = "LSG-451";

// device VIDs are all the same
unsigned short dev1VID = 0x041f;
unsigned short dev2VID = 0x041f;
unsigned short dev3VID = 0x041f;
unsigned short dev4VID = 0x041f;
unsigned short dev5VID = 0x041f;
unsigned short dev6VID = 0x041f;
unsigned short dev7VID = 0x041f;

// each device has its own pid
unsigned short dev1PID = 0x1201; // device PID for Vaunix LSG-402
unsigned short dev2PID = 0x1202; // device PID for Vaunix LSG-152
unsigned short dev3PID = 0x1203; // device PID for Vaunix LSG-251
unsigned short dev4PID = 0x1204; // device PID for Vaunix LSG-602
unsigned short dev5PID = 0x1205; // device PID for Vaunix LSG-222
unsigned short dev6PID = 0x1206; // device PID for Vaunix LSG-121
unsigned short dev7PID = 0x1209; // device PID for Vaunix LSG-451

/* stuff for the threads */
pthread_t threads[MAXDEVICES];
usb_dev_handle *thread_devhandles[MAXDEVICES];
#define THREAD_IDLE 0
#define THREAD_START 1
#define THREAD_WRITE 2
#define THREAD_EXIT 3
#define THREAD_DEAD 4
#define THREAD_ERROR -1

// --------------------- other device specific equates ------------------------
#define HW_MAXP 40; // MaxPower is the output of the device in db -- +10 in this case
                    // NB -- the value used in the SetPower function is relative attenuation
                    // not absolute power!!

// --------------- Device IO support functions ----------------------------

bool CheckDeviceOpen(DEVID deviceID) {
  if (TestMode) return TRUE;	// in test mode all devices are always available
  if ((lsg[deviceID].DevStatus & DEV_OPENED) && (deviceID != 0))
    return TRUE;
  else
    return FALSE;
}

// ------------------------------------------------------------------------
bool DevNotLocked(DEVID deviceID) {
  if (TestMode) return TRUE;	// this shouldn't happen, but just in case...
  if (!(lsg[deviceID].DevStatus & DEV_LOCKED))
    return TRUE;				// we return TRUE if the device is not locked!
  else
    return FALSE;
}

// ------------------------------------------------------------------------
void LockDev(DEVID deviceID, bool lock) {
  if (TestMode) return;			// this shouldn't happen, but just in case...
  if (lock) {
    lsg[deviceID].DevStatus = lsg[deviceID].DevStatus | DEV_LOCKED;
    return;
  } else {
    lsg[deviceID].DevStatus = lsg[deviceID].DevStatus & ~DEV_LOCKED;
    return;
  }
}

/* A function to display the status as string */
char* fnLSG_perror(LVSTATUS status) {
  strcpy(errmsg, "STATUS_OK");
  if (BAD_PARAMETER == status) strcpy(errmsg, "BAD_PARAMETER");
  if (BAD_HID_IO == status) strcpy(errmsg, "BAD_HID_IO");
  if (DEVICE_NOT_READY == status) strcpy(errmsg, "DEVICE_NOT_READY");
  
  // Status returns for DevStatus
  if (INVALID_DEVID == status) strcpy(errmsg, "INVALID_DEVID");
  if (DEV_CONNECTED == status) strcpy(errmsg, "DEV_CONNECTED");
  if (DEV_OPENED == status) strcpy(errmsg, "DEV_OPENED");
  if (SWP_ACTIVE == status) strcpy(errmsg,  "SWP_ACTIVE");
  if (SWP_UP == status) strcpy(errmsg, "SWP_UP");
  if (SWP_REPEAT == status) strcpy(errmsg, "SWP_REPEAT");
  if (SWP_BIDIRECTIONAL == status) strcpy(errmsg, "SWP_BIDIRECTIONAL");

// 		These values aren't returned in status
// 		(Internal values in DevStatus)
//		#define DEV_LOCKED   0x00002000 	// set if we don't want read thread updates of the device parameters
//		#define DEV_RDTHREAD   0x00004000 	// set when the read thread is running


  return errmsg;

}

// -- allow applications to query the library version --
char LibVersion[] = LIBVER;
char* fnLSG_LibVersion(void) {
  return LibVersion;
}
  

/* functions based on hid_io.cpp */
bool VNXOpenDevice(DEVID deviceID) {

  if (!(lsg[deviceID].DevStatus & DEV_CONNECTED))	// we can't open a device that isn't there!
    return DEVICE_NOT_READY;
  
  if (DEBUG_OUT > 1) printf("Starting thread...\r\n");
  lsg[deviceID].thread_command = THREAD_START; /* open device and start processing */
  pthread_create(&threads[deviceID], NULL, brick_handler_function, (void*)deviceID);
  //  sleep(3);
  lsg[deviceID].DevStatus = lsg[deviceID].DevStatus | DEV_OPENED;
  
  return STATUS_OK;
}

void report_data_decode(unsigned char rcvdata[], int tid) {
  int i;
  unsigned long dataval;
  char temp[32];

  if (DEBUG_OUT > 1) {
    printf("Decoding ");
    for (i=0; i<8; i++)
      printf("%02x ", rcvdata[i]);
    printf("\r\n");
  }

  /* the first byte tells us the type, the second is the data length
     tid is the thread ID it came from so we can stash the value into lsg[] */
  /* first decode the bytes */
  dataval = 0;
  if (0 < rcvdata[1]) {
    for (i=0; i<rcvdata[1]; i++)
      dataval = (dataval<<8) + rcvdata[1+rcvdata[1]-i];
  }
  if (DEBUG_OUT > 1) printf("Data payload decodes to %ld (%08x)\r\n", dataval, dataval);
  /* now we'll assign it to lsg[] */

  // handle the status report
  switch(rcvdata[0]) {
  case VNX_STATUS:
    if (DevNotLocked(tid)) {
      lsg[tid].Frequency = rcvdata[2] + (rcvdata[3]<<8) + (rcvdata[4]<<16) + (rcvdata[5]<<24); // update the frequency
      if (DEBUG_OUT > 0) printf("Status decode device = %d, Frequency = %d", tid, lsg[tid].Frequency);
      if (DEBUG_OUT > 1) printf("."); fflush(0);

      if (rcvdata[6] & (SWP_ONCE | SWP_CONTINUOUS))		// are we sweeping?
		lsg[tid].DevStatus = lsg[tid].DevStatus | SWP_ACTIVE;
      else
		lsg[tid].DevStatus = lsg[tid].DevStatus & ~SWP_ACTIVE;

      // -- fill in the SWP_UP status bit
      if (rcvdata[6] & (SWP_DIRECTION))		// are we downwards?
		lsg[tid].DevStatus = lsg[tid].DevStatus & ~SWP_UP;
      else
		lsg[tid].DevStatus = lsg[tid].DevStatus | SWP_UP;

      // -- fill in the continuous sweep bit
      if (rcvdata[6] & (SWP_CONTINUOUS))		// are we in continuous sweep mode
		lsg[tid].DevStatus = lsg[tid].DevStatus | SWP_REPEAT;
      else
		lsg[tid].DevStatus = lsg[tid].DevStatus & ~SWP_REPEAT;
		

    } /* if devnotlocked() */

    if (DEBUG_OUT > 0) printf("DevStatus at the end of ParseHidPacket: %d \n", (lsg[tid].DevStatus & DEV_RDTHREAD));
    break;

  case VNX_FREQUENCY:
    if (DEBUG_OUT > 0) printf(" Frequency = %d\n", dataval);
    if (DevNotLocked(tid))
      lsg[tid].Frequency = dataval;
    break;

  case VNX_FDWELL:
    if (DEBUG_OUT > 0) printf(" Dwell Time = %d\n", dataval);
    if (DevNotLocked(tid))
      lsg[tid].DwellTime = dataval;
    break;

  case VNX_FSTART:
    if (DEBUG_OUT > 0) printf(" Sweep Start Frequency = %d\n", dataval);
    if (DevNotLocked(tid))
      lsg[tid].StartFrequency = dataval;
    break;

  case VNX_FSTOP:
    if (DEBUG_OUT > 0) printf(" Sweep End Frequency = %d\n", dataval);
    if (DevNotLocked(tid))
      lsg[tid].EndFrequency = dataval;
    break;

  case VNX_SWEEP:
    if (DEBUG_OUT > 0) printf(" Sweep Mode = %d\n", rcvdata[2]);
    if (DevNotLocked(tid))
      lsg[tid].Frequency = dataval;
    break;

  case VNX_FSTEP:
    if (DEBUG_OUT > 0) printf(" Frequency Step = %d\n", dataval);
    if (DevNotLocked(tid))
      lsg[tid].FrequencyStep = dataval;
    break;

  case VNX_RFMUTE:
    if (rcvdata[2]) strcpy(temp, "RF ON");
    else strcpy(temp, "RF OFF");
    if (DEBUG_OUT > 0) printf("%s \n", temp);

    if (DevNotLocked(tid)) {
      if (rcvdata[2]) {
	lsg[tid].Modebits = lsg[tid].Modebits | MODE_RFON;
      } else {
	lsg[tid].Modebits = lsg[tid].Modebits & ~MODE_RFON;
      }
    }
    break;

  case VNX_INTOSC:
    if (rcvdata[2]) strcpy(temp, "Using Internal Osc");
    else strcpy(temp, "Using External Osc");
    if (DEBUG_OUT > 0) printf("%s \n", temp);

    if (DevNotLocked(tid)) {
      if (rcvdata[2]) {
	lsg[tid].Modebits = lsg[tid].Modebits | MODE_INTREF;
      } else {
	lsg[tid].Modebits = lsg[tid].Modebits & ~MODE_INTREF;
      }
    }
    break;

  case VNX_PWR:
    if (DEBUG_OUT > 0) printf(" Power Setting = %d units)\n", rcvdata[2]);
    if (DevNotLocked(tid))
      lsg[tid].Power = rcvdata[2];
    break;

  case VNX_MAX_PWR:
    if (DEBUG_OUT > 0) printf(" Max Power Setting = %d units)\n", rcvdata[2]);
    if (DevNotLocked(tid))
      lsg[tid].MaxPower = rcvdata[2];
    break;

  case VNX_MINFREQUENCY:
    if (DEBUG_OUT > 0) printf(" Minimum Frequency = %d\n", dataval);
    if (DevNotLocked(tid))
      lsg[tid].MinFrequency = dataval;	// our device returns MinFrequency in 100 KHz units
    break;

  case VNX_MAXFREQUENCY:
    if (DEBUG_OUT > 0) printf(" Maximum Frequency = %d\n", dataval);
    if (DevNotLocked(tid))
      lsg[tid].MaxFrequency = dataval;	// same as MinFrequency
    break;

  case VNX_SETSERNUM:
    if (DEBUG_OUT > 0) printf(" Serial Number = %d\n", dataval);
    if (DevNotLocked(tid))
      lsg[tid].SerialNumber = dataval;		// NB -- we never use this path!
    break;
  } /* switch */

  return;
}

// ************* The read thread handler for the brick ***********************
void *brick_handler_function (void *threadID) {
  int i, tid;
  tid = (int)threadID;
  struct usb_bus *busses;
  struct usb_bus *bus;
  unsigned char sendbuff[8];
  unsigned char rcvbuff[32];
  int usb_status;
  char fullpath[128];
  int retries;
 
  if (DEBUG_OUT > 0) printf("Starting thread for device %d\r\n", tid);
  while ((lsg[tid].thread_command >=0) &&
	 (lsg[tid].thread_command != THREAD_EXIT)) {
    switch(lsg[tid].thread_command) {
    case THREAD_IDLE: /* idle */
      /* this is where we wait for incoming USB data */
      //      printf("Starting a read...\r\n");
      /* If someone is writing to the device, don't try to read from it */
      if (lsg[tid].pending) {
	if (DEBUG_OUT > 1) {printf("z"); fflush(0);}
	usleep(10000); /* wait 10ms for whatever to finish */
	goto TIDLE_DONE;
      }
      usb_status = -1;
      retries = 50;
      while ((usb_status < 0) && (retries--) && (THREAD_IDLE == lsg[tid].thread_command)) {
	usb_status = usb_interrupt_read(thread_devhandles[tid],  // handle
					ENDPOINT_INT_IN, // endpoint
					rcvbuff, 		 // buffer
					PACKET_INT_LEN,  // max length
					TIMEOUT);
	if (usb_status < 0) usleep(1000); /* wait 20 ms before trying again */
      }
      //      printf("Thread %d reports %d...\r\n", tid, usb_status);
      if (usb_status >= 0) {
	if (DEBUG_OUT > 1) {
	  printf("Thread %d reports %d...", tid, usb_status);
	  for (i=0; i<usb_status; i++)
	    printf("%02x ", rcvbuff[i]);
	  printf("\r\n");
	}
	/* decode the HID data */
	report_data_decode(rcvbuff, tid);
      } else
	if (DEBUG_OUT > 0) perror("THREAD_IDLE");
    TIDLE_DONE:
      usleep(8000);  // 8 ms
      //            sleep(1);
      break;
    case THREAD_START: /* starting up */
      /* we'll open the device. First we have to locate it */
      if (DEBUG_OUT > 0) printf("Thread %d is looking for the device\r\n", tid);

      usb_find_busses();
      usb_find_devices();
      busses = usb_get_busses();

      lsg[tid].thread_command = THREAD_ERROR; /* assume it will fail */
      for (bus = busses; bus; bus = bus->next) {
	if (THREAD_IDLE == lsg[tid].thread_command) break;
	struct usb_device *dev;

	for (dev = bus->devices; dev; dev = dev->next) {
	  if (THREAD_IDLE == lsg[tid].thread_command) break;
	  if (DEBUG_OUT > 1) printf("Thread %d sez- Vendor: %04x PID: %04x\r\n", tid, dev->descriptor.idVendor, dev->descriptor.idProduct);
	  thread_devhandles[tid] = usb_open(dev);
	  usb_status = usb_get_string_simple(thread_devhandles[tid], dev->descriptor.iSerialNumber, rcvbuff, sizeof(rcvbuff));
	  if (DEBUG_OUT > 1) printf("string %d = [%s] looking to match [%s]\r\n", dev->descriptor.iSerialNumber, rcvbuff, lsg[tid].Serialstr);
	  //	  usb_close(thread_devhandles[tid]);
	  if ((dev->descriptor.idVendor == lsg[tid].idVendor) &&
	      (dev->descriptor.idProduct == lsg[tid].idProduct) &&
	      (0 == strcmp(rcvbuff, lsg[tid].Serialstr))) {
	    /* we found the device. We'll open it */
	    if (DEBUG_OUT > 1) printf("Opening file [%s]\r\n", dev->filename);
	    thread_devhandles[tid] = usb_open(dev);

	    usb_detach_kernel_driver_np(thread_devhandles[tid], 0);

	    usb_status = usb_set_configuration (thread_devhandles[tid], 1);
	    if (DEBUG_OUT > 1) printf ("set configuration: %s\n", usb_status ? "failed" : "passed");

	    usb_status = usb_claim_interface (thread_devhandles[tid], 0);
	    if (DEBUG_OUT > 1) printf ("claim interface: %s\n", usb_status ? "failed" : "passed");

	    lsg[tid].thread_command = THREAD_IDLE;
	    break;
	  } //else
	  usb_close(thread_devhandles[tid]);
	}
      }
      //      lsg[tid].thread_command = THREAD_IDLE;
      break;
    case THREAD_WRITE:
#if 0 // we don't use the threaded write scheme...
      /* we have data to write to the device */
      printf("Thread %d: writing to handle %d ", tid, thread_filehandles[tid]);
      for (i=0; i<8; i++) {
	printf("%02x ", lsg[tid].outbuff[i]);
      }
      //      usb_status = write(thread_filehandles[tid], lsg[tid].outbuff, 8);
      //      printf("(status=%d)\r\n", usb_status);
      lsg[tid].thread_command = THREAD_IDLE;
#endif
      break;
    } /* switch */
  } /* while */
  if (DEBUG_OUT > 0) printf("Exiting thread for device %d\r\n", tid);
  if (THREAD_EXIT == lsg[tid].thread_command)
    usb_close(thread_devhandles[tid]);
  lsg[tid].thread_command = THREAD_DEAD;
  pthread_exit(NULL);
  
}

// -------------- SendReport -------------------------------------------------

bool SendReport(int deviceID, char command, char *pBuffer, int cbBuffer)
{
  int i;
  int send_status;
  int retries;
  // Make sure the buffer that is being passed to us fits
  if (cbBuffer > HR_BLOCKSIZE) {
    // Report too big, don't send!
    return FALSE;
  }

  char Report[8];
  
  if (DEBUG_OUT > 1) printf("SR: command=%x cbBuffer=%x\r\n", command, cbBuffer);
  lsg[deviceID].outbuff[0] = command;		// command to device
  lsg[deviceID].outbuff[1] = cbBuffer;
  lsg[deviceID].outbuff[2] = pBuffer[0];
  lsg[deviceID].outbuff[3] = pBuffer[1];
  lsg[deviceID].outbuff[4] = pBuffer[2];
  lsg[deviceID].outbuff[5] = pBuffer[3];
  lsg[deviceID].outbuff[6] = pBuffer[4];
  lsg[deviceID].outbuff[7] = pBuffer[4];
  if (DEBUG_OUT > 1) {
    printf("SR: ");
    for (i=0; i<8; i++) {
      printf("%02x ", lsg[deviceID].outbuff[i]);
    }
    printf("\r\n");
  }

  /* we have to wait for a file handle to appear */
  retries = 0;
  while ((0 == thread_devhandles[deviceID]) && (retries++ < 10))
    sleep(1);
  /* we have data to write to the device */
  if (DEBUG_OUT > 1) printf("SR: sending the write...\r\n");
  send_status = usb_control_msg(thread_devhandles[deviceID],
				0x21,
				0x09, //HID_REPORT_SET,
				0x200,
				0,
				lsg[deviceID].outbuff,
				PACKET_CTRL_LEN,
				TIMEOUT);

  if (DEBUG_OUT > 1) {
    printf("(status=%d handle=%d)", send_status, thread_devhandles[deviceID]);
    if (send_status < 0) perror("SendReport"); else printf("\r\n");
  }
  //lsg[tid].thread_command = THREAD_IDLE;


#if 0
  lsg[deviceID].thread_command = THREAD_WRITE;
#endif
  // We should have transferred exactly as many bytes as we sent
  // assert(cbTransferred == sOutReportSize);
  
  // printf(" sending command: %02x of %d bytes\n", command, cbTransferred );
  
  return TRUE;
}

// ------------ GetParameter ---------------------------------------------
//
// The GetParam argument is the command byte sent to the device to get
// a particular value. The response is picked up by the read thread and
// parsed by it. The parser clears the corresponding event.


bool GetParameter(int deviceID, int GetParam)
{
	char VNX_param[4] = {0, 0, 0, 0};

	lsg[deviceID].pending = 1;
	usleep(10000); /* allow 10 ms for any read activity to stop */
	if (DEBUG_OUT > 0) printf(" sending a GET command = %x\n", (char) GetParam );
	if (!SendReport(deviceID, (char)GetParam, VNX_param, 0)) {
	  lsg[deviceID].pending = 0;
	  return FALSE;
	}

	if (DEBUG_OUT > 0) printf(" SendReport sent a GET command successfully = %x\n", (char) GetParam );
	lsg[deviceID].pending = 0;
	return TRUE;
}

// -------------- Get Routines to read device settings --------------------
//
// Note: for these functions deviceID is not checked for validity
//		 since it was already checked in the calling program.

bool GetFrequency(DEVID deviceID) {
  if (DEBUG_OUT > 0) printf(" In GetFrequency\n");
  // --- first we send the command out to the device --
  if (!GetParameter(deviceID, VNX_FREQUENCY))
    return FALSE;
  return TRUE;
}

// -------------------------------

bool GetPower(DEVID deviceID) {
  // --- first we send the command out to the device --
  if (!GetParameter(deviceID, VNX_PWR))
    return FALSE;
  return TRUE;
}

// -------------------------------

bool GetFStart(DEVID deviceID) {
  // --- first we send the command out to the device --
  if (!GetParameter(deviceID, VNX_FSTART))
    return FALSE;
  return TRUE;
}

// -------------------------------

bool GetFEnd(DEVID deviceID) {	
  // --- first we send the command out to the device --
  if (!GetParameter(deviceID, VNX_FSTOP))
    return FALSE;
  return TRUE;
}

// -------------------------------
bool GetDwell(DEVID deviceID) {
  // --- first we send the command out to the device --
  if (!GetParameter(deviceID, VNX_FDWELL))
    return FALSE;
  return TRUE;
}

// -------------------------------
bool GetMaxPower(DEVID deviceID) {
  // --- first we send the command out to the device --
  if (!GetParameter(deviceID, VNX_MAX_PWR))
    return FALSE;
  return TRUE;
}
// -------------------------------
bool GetRF_On(DEVID deviceID) {
  // --- first we send the command out to the device --
  if (!GetParameter(deviceID, VNX_RFMUTE))
    return FALSE;
  return TRUE;
}

// ---------------------------------
bool GetUseIntOsc(DEVID deviceID) {
  // --- first we send the command out to the device --
  if (!GetParameter(deviceID, VNX_INTOSC))
    return FALSE;
  return TRUE;
}

bool GetFMinimum(DEVID deviceID) {
  // --- first we send the command out to the device --
  if (!GetParameter(deviceID, VNX_MINFREQUENCY))
    return FALSE;
  return TRUE;
}

// -------------------------------
bool GetFMaximum(DEVID deviceID) {
  // --- first we send the command out to the device --
  if (!GetParameter(deviceID, VNX_MAXFREQUENCY))
    return FALSE;
  return TRUE;
}

// -------------------------------
bool GetFrequencyStep(DEVID deviceID) {	
  // --- first we send the command out to the device --
  if (!GetParameter(deviceID, VNX_FSTEP))
    return FALSE;
  return TRUE;
}


/* functions to manage devices, not getting or retrieving data */
/*-------------------------------------------------------------*/

void FindVNXDevices()
{
  //  int i;			// general purpose temp variable
  bool bFound;
  int hTemp;  			// temporary variable
  int HWType; 			// temporary variable for hardware/model type
  int HWMinFrequency; 	// temporary variable for default minimum frequency
  int HWMaxFrequency; 	// temporary variable for default maximum frequency
  int HWMaxPower; 		// temporary variable for default maximum power
  int HWMinPower;		// temporary variable for default minimum power
  char HWName[32];  	// temporary variable for the hardware model name
  char HWSerial[8]; 	// temporary holder for the serial number
  struct usb_dev_handle *devhandle;
  struct usb_bus *busses;
  char sendbuff[8];
  char rcvbuff[32];
  int usb_status;
    
  usb_init();
  if (DEBUG_OUT > 2)
    usb_set_debug(3); /* if we want lots of debug, let's see the USB output too. */
  else
    usb_set_debug(0);
  usb_find_busses();
  usb_find_devices();
  
  busses = usb_get_busses();
        
  struct usb_bus *bus;
  int c, i, a;
  int send_status, open_status;
  
  /* ... */
    
  // We need to remove devices from our table that are no longer connected ---
  // to do this we clear the "connected" flag for each table entry that is not open initially
  // then, as we find them we re-set the "connected" flag
  // anybody who doesn't have his "connected" flag set at the end is gone - we found it
  // previously but not this time
  
  for (i = 1; i<MAXDEVICES; i++){
    if ((lsg[i].SerialNumber != 0) && !(lsg[i].DevStatus & DEV_OPENED))
      lsg[i].DevStatus = lsg[i].DevStatus & ~DEV_CONNECTED; 	
  }

  for (bus = busses; bus; bus = bus->next) {
    struct usb_device *dev;
    
    for (dev = bus->devices; dev; dev = dev->next) {
      if (DEBUG_OUT > 1) printf("Vendor: %04x PID: %04x\r\n", dev->descriptor.idVendor, dev->descriptor.idProduct);
      HWType = 0;
      /* check this device to see if it's one of our devices */
      if ((dev1VID == dev->descriptor.idVendor) &&
	  (dev1PID == dev->descriptor.idProduct)) HWType = 1;
      if ((dev2VID == dev->descriptor.idVendor) &&
	  (dev2PID == dev->descriptor.idProduct)) HWType = 2;
      if ((dev3VID == dev->descriptor.idVendor) &&
	  (dev3PID == dev->descriptor.idProduct)) HWType = 3;
      if ((dev4VID == dev->descriptor.idVendor) &&
	  (dev4PID == dev->descriptor.idProduct)) HWType = 4;
	  if ((dev5VID == dev->descriptor.idVendor) &&
	  (dev5PID == dev->descriptor.idProduct)) HWType = 5;
      if ((dev6VID == dev->descriptor.idVendor) &&
	  (dev6PID == dev->descriptor.idProduct)) HWType = 6;
      if ((dev7VID == dev->descriptor.idVendor) &&
	  (dev7PID == dev->descriptor.idProduct)) HWType = 7;
	  
	  
	  
	  
      if (HWType) { /* we like this device and we'll keep it */
	if (DEBUG_OUT > 1) printf("Opening device %04x:%04x serial %04x type %d\r\n",
			      dev->descriptor.idVendor,
			      dev->descriptor.idProduct,
			      dev->descriptor.iSerialNumber, HWType);
	devhandle = usb_open(dev);
	if (DEBUG_OUT > 1)  printf("LSG device found @ address [%s]\r\n", dev->filename);
	usb_status = usb_get_string_simple(devhandle, dev->descriptor.iSerialNumber, rcvbuff, sizeof(rcvbuff));
	if (DEBUG_OUT > 1) printf("string %d = [%s]\r\n", dev->descriptor.iSerialNumber, rcvbuff);
	if (usb_status < 0) strcpy(HWSerial, ""); else strcpy(HWSerial, rcvbuff+3);
	usb_close(devhandle);
	switch(HWType) {
	case 1: // LSG-402
	  strcpy(HWName, sVNX1);
	  HWMinFrequency = 10000;			// 1 Ghz in 100Khz units
	  HWMaxFrequency = 40000;			// 4 Ghz in 100Khz units
	  HWMaxPower = HW_MAXP;				// 10db in our .25db step system
	  HWMinPower = HW_MAXP - 220;		// the '402 has a 55 db attenuation range
	  break;
	case 2:  // LSG-152
	  strcpy(HWName, sVNX2);
	  HWMinFrequency = 2500;			// 250 MHz in 100Khz units
	  HWMaxFrequency = 15000;			// 1.5 Ghz in 100Khz units
	  HWMaxPower = HW_MAXP;
	  HWMinPower = HW_MAXP - 220;		// the '152 has a 55 db attenuation range
	  break;
	case 3: // LSG-251
	  strcpy(HWName, sVNX3);
	  HWMinFrequency = 500;				// 50 MHz in 100KHz units
	  HWMaxFrequency = 2500;			// 250 MHz in 100KHz units
	  HWMaxPower = HW_MAXP;
	  HWMinPower = HW_MAXP - 220;		// the '251 has a 55 db attenuation range
	  break;
	case 4: // LSG-602
	  strcpy(HWName, sVNX4);
	  HWMinFrequency = 15000;			// 1 Ghz in 100Khz units
	  HWMaxFrequency = 60000;			// 6 Ghz in 100Khz units
	  HWMaxPower = HW_MAXP;
	  HWMinPower = HW_MAXP - 200;		// the '602 has 50 db attenuation
	  break;
	case 5: // LSG-222
	  strcpy(HWName, sVNX4);
	  HWMinFrequency = 5000;			// 500 MHz in 100KHz units
	  HWMaxFrequency = 22000;			// 2.2 GHz in 100KHz units
	  HWMaxPower = HW_MAXP;
	  HWMinPower = HW_MAXP - 220;		// the '222 has a 55 db attenuation range
	  break; 
	case 6: // LSG-121
	  strcpy(HWName, sVNX4);
	  HWMinFrequency = 200;				// 20 MHz in 100KHz units
	  HWMaxFrequency = 1200;			// 120 MHz in 100KHz units
	  HWMaxPower = HW_MAXP;
	  HWMinPower = HW_MAXP - 220;		// the '121 has a 55 db attenuation range
	  break; 	  
	case 7: // LSG-451
	  strcpy(HWName, sVNX4);
	  HWMinFrequency = 700;				// 70 MHz in 100KHz units
	  HWMaxFrequency = 4500;			// 450 MHz in 100KHz units
	  HWMaxPower = HW_MAXP;
	  HWMinPower = HW_MAXP - 220;		// the '451 has a 55 db attenuation range
	  break; 	  
	  
	  
	} /* HWType switch */
	
	/* find an open slot to save the data */
	// lets see if we have this unit in our table of devices already
	bFound = FALSE;
	
	for (i = 1; i<MAXDEVICES; i++){
	  if (lsg[i].SerialNumber == atoi(HWSerial)) {
	    // we already have the device in our table
	    bFound = TRUE;
	    lsg[i].DevStatus = lsg[i].DevStatus | DEV_CONNECTED; // its here, mark it as connected
	    // at this point the device is present, but not in use, no sense looking more
	    break;
	  }
	}	// end of for loop
	
	// if the device isn't in the table we need to add it
	if (!bFound) {
	  hTemp = 0;
	  for (i = 1; i<MAXDEVICES; i++) {
	    if (lsg[i].SerialNumber == 0) {
	      hTemp = i;
	      break;
	    }
	  } // end of for loop search for an empty slot in our array of devices
	  
	  /* save all of the data we've already aquired */
	  if (hTemp) {
	    lsg[hTemp].SerialNumber = atoi(HWSerial);		            	// save the device's serial number
	    lsg[hTemp].DevStatus = lsg[hTemp].DevStatus | DEV_CONNECTED;    // mark it as present
	    strcpy (lsg[hTemp].ModelName, HWName);     		    			// save the device's model name
	    //	    lsg[hTemp].usb_device_handle = NULL;     				// make sure we're in the unopened state!
	    lsg[hTemp].MinFrequency = HWMinFrequency;
	    lsg[hTemp].MaxFrequency = HWMaxFrequency;			    		// default values for frequency range
	    lsg[hTemp].MaxPower = HWMaxPower;				    			// default value for maximum power
	    lsg[hTemp].MinPower = HWMinPower;								// and for minimum power
		
		/* The device has been closed so let's make sure we can find it again */
	    lsg[hTemp].idVendor = dev->descriptor.idVendor;
	    lsg[hTemp].idProduct = dev->descriptor.idProduct;
	    strcpy(lsg[hTemp].Serialstr, rcvbuff);
	    if (DEBUG_OUT > 1) {
	      printf("Stored as new device #%d\r\n", hTemp);
	      printf("Serial number=%d\r\n", lsg[hTemp].SerialNumber);
	      printf("Devstatus=%08x\r\n", lsg[hTemp].DevStatus);
	      printf("Model name=%s\r\n", lsg[hTemp].ModelName);
	      printf("MinFrequency=%d\r\n", lsg[hTemp].MinFrequency);
	      printf("MaxFrequency=%d\r\n", lsg[hTemp].MaxFrequency);
	      printf("MaxPower=%d\r\n", lsg[hTemp].MaxPower);
		  printf("MinPower=%d\r\n", lsg[hTemp].MinPower);
	      printf("Vendor ID=%04x\r\n", lsg[hTemp].idVendor);
	      printf("Product ID=%04x\r\n", lsg[hTemp].idProduct);
	      printf("Serial number=%s\r\n", lsg[hTemp].Serialstr);
	    }
	  } else {
	    // our table of devices is full, not much we can do
	  }
	} /* if !bfound  */
	/* get any other data we might need */
      } /* if HWType */
    } /* for dev */
  } /* for bus */

  /* clean up the structure and mark unused slots */
  for (i = 1; i<MAXDEVICES; i++){
    if ((lsg[i].SerialNumber != 0) && !(lsg[i].DevStatus & DEV_CONNECTED))
      lsg[i].SerialNumber = 0;	// mark this slot as unused 	

    if (lsg[i].SerialNumber == 0)
      lsg[i].DevStatus = 0;		// clear the status for robustness!
  }	// end of zombie removal for loop
}

/* ----------------------------------------------------------------- */

void fnLSG_Init(void) {
  /* clear out the storage structure. Must be called once before anything else */
  int i;
  int status;

  for (i = 0; i<MAXDEVICES; i++){
    lsg[i].DevStatus = 0;		// init to no devices connected
    lsg[i].SerialNumber = 0;	// clear the serial number
    lsg[i].ModelName[0] = 0;	// put a null string in each model name field
  }

  usb_init();
  if (DEBUG_OUT > 0)  printf("library version %s\r\n", fnLSG_LibVersion());
}

void fnLSG_SetTestMode(bool testmode) {
  TestMode = testmode;
}

int fnLSG_GetNumDevices() {
  int retval = 0;
  int NumDevices = 0;
  int i;
  
  // See how many devices we can find, or have found before
  if (TestMode){
    
    // construct a fake device
    lsg[1].SerialNumber = 123402;
    lsg[1].DevStatus = lsg[1].DevStatus | DEV_CONNECTED;
    lsg[1].MinFrequency = 10000;		// 1 Ghz in 100KHz resolution
    lsg[1].MaxFrequency = 40000;		// 4 Ghz in 100KHz resolution
    lsg[1].MaxPower = 40;				// 10db in our .25db step system
    lsg[1].MinPower = 40 - 220;			// the LSG-402 has a 55 db attenuation range
    strcpy (lsg[1].ModelName, "LSG-402");
    
    // construct a second fake device
    lsg[2].SerialNumber = 456602;
    lsg[2].DevStatus = lsg[2].DevStatus | DEV_CONNECTED;
    lsg[2].MinFrequency = 15000;		// 1.5 Ghz in 100Khz resolution
    lsg[2].MaxFrequency = 60000;		// 6 Ghz in 100Khz resolution
    lsg[2].MaxPower = 40;				// 10db in our .25db step system
    lsg[2].MinPower = 40 - 200;			// the '602 has a 50 db attenuation range
    strcpy (lsg[2].ModelName, "LSG-602");

    retval = 2;
    
  } else {
    // go look for some real hardware
    FindVNXDevices();

    // Total up the number of devices we have
    for (i = 0; i < MAXDEVICES; i++){
      if (lsg[i].DevStatus & DEV_CONNECTED) NumDevices++; 
    }
    retval = NumDevices;

  }
  return retval;
}

int fnLSG_GetDevInfo(DEVID *ActiveDevices) {
  int i;
  int NumDevices = 0;
  
  if ( ActiveDevices == NULL) return 0;	// bad array pointer, no place to put the DEVIDs
  
  for (i = 1; i < MAXDEVICES; i++){ 	// NB -- we never put an active device in lsg[0] - so DEVIDs start
										// at 1
    if (lsg[i].DevStatus & DEV_CONNECTED) {
      ActiveDevices[NumDevices] = i;
      NumDevices++;
    }
  }
  
  return NumDevices;
}

int fnLSG_GetModelName(DEVID deviceID, char *ModelName) {
  int NumChars = 0;

  if (deviceID >= MAXDEVICES){
    return 0;
  }

  NumChars = strlen(lsg[deviceID].ModelName);
  // If NULL result pointer, just return the number of chars in the name
  if ( ModelName == NULL) return NumChars;
  strcpy(ModelName, lsg[deviceID].ModelName);

  return NumChars;
}

int fnLSG_InitDevice(DEVID deviceID) {

  if ((deviceID >= MAXDEVICES) || (deviceID == 0)){
    return INVALID_DEVID;
  }
  
  if (TestMode)
    lsg[deviceID].DevStatus = lsg[deviceID].DevStatus | DEV_OPENED;
  else {
    // Go ahead and open a handle to the hardware
    if (VNXOpenDevice(deviceID))//VNXOpenDevice returns 0 if the open succeeded
      return DEVICE_NOT_READY;
    if (DEBUG_OUT > 0) printf("Time to start getting device parameters...");

      // Get the rest of the device parameters from the device
	  // Note - some were already set as defaults based on product ID
    
    if (!GetFrequency(deviceID))	// read the frequency from the device (in 100KHz units)
      return BAD_HID_IO;
  
    if (DEBUG_OUT > 0) printf("Got Frequency...");

    if (!GetFStart(deviceID))		// read the sweep start frequency
      return BAD_HID_IO;
    if (DEBUG_OUT > 0) printf("Got FStart...");

    if (!GetFEnd(deviceID))			// read the sweep end frequency
      return BAD_HID_IO;
      if (DEBUG_OUT > 0) printf("Got FEnd...");

    if (!GetDwell(deviceID))		// read the sweep dwell time
      return BAD_HID_IO;
      if (DEBUG_OUT > 0) printf("Got DwellTime...");
	  
    if (!GetFrequencyStep(deviceID)) 
      return BAD_HID_IO;
      if (DEBUG_OUT > 0) printf("Got Frequency Step...");

    if (!GetPower(deviceID))
      return BAD_HID_IO;
      if (DEBUG_OUT > 0) printf("Got Power...");

    if (!GetMaxPower(deviceID))
      return BAD_HID_IO;
    if (DEBUG_OUT > 0) printf("Got Max Power...");

    if (!GetRF_On(deviceID))
      return BAD_HID_IO;
    if (DEBUG_OUT > 0) printf("Got RF On...");

    if (!GetUseIntOsc(deviceID))
      return BAD_HID_IO;
    
    if (!GetFMinimum(deviceID))
      return BAD_HID_IO;
    if (DEBUG_OUT > 0) printf("Got FMinimum...");

    if (DEBUG_OUT > 0) printf("Device Open Successful!\r\n");
  } // end of real device open process case

  // if we got here everything worked OK
  return STATUS_OK;
}

int fnLSG_CloseDevice(DEVID deviceID) {
  
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  if (TestMode)
    lsg[deviceID].DevStatus = lsg[deviceID].DevStatus & ~DEV_OPENED;
  else {

    // Go ahead and close this hardware - the first step is to stop its read thread
    lsg[deviceID].thread_command = THREAD_EXIT;
    
    // The thread handler will close the device. We'll wait up to 1 second then give up.
    int retries;
    retries = 10;
    while (retries && (lsg[deviceID].thread_command != THREAD_DEAD)) {
      usleep(100000); /* wait 100 ms */
      retries--;
    }
    if (DEBUG_OUT > 0) printf("After telling the thread to close, we have thread_command=%d retries=%d\r\n", lsg[deviceID].thread_command, retries);
    //    threads[deviceID] = NULL;
    lsg[deviceID].thread_command = THREAD_IDLE;

    // Mark it closed in our list of devices
    lsg[deviceID].DevStatus = lsg[deviceID].DevStatus & ~DEV_OPENED;
  }

  return STATUS_OK;

}

int fnLSG_GetSerialNumber(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return 0;
  
  return lsg[deviceID].SerialNumber;
}

LVSTATUS fnLSG_SetFrequency(DEVID deviceID, int frequency) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  LockDev(deviceID, TRUE); // prevent updates to the frequency variable reported by the device!
  
  int old_frequency = lsg[deviceID].Frequency;
  
  if ((frequency >= lsg[deviceID].MinFrequency) && (frequency <= lsg[deviceID].MaxFrequency)) {
    lsg[deviceID].Frequency = frequency;
    if (TestMode)
      return STATUS_OK;// in test mode we update our internal variables, but don't talk to the real HW
  } else {
    return BAD_PARAMETER;
  }
  // the frequency value is OK, lets send it to the hardware
  unsigned char *ptr = (unsigned char *) &lsg[deviceID].Frequency;

  if (DEBUG_OUT > 0) printf("deviceID = %d ptr = %x Frequency = %x\r\n", deviceID , ptr, *ptr);

  lsg[deviceID].pending = 1;
  usleep(10000); /* wait 10 ms for any reads to stop */
  if (!SendReport(deviceID, VNX_FREQUENCY | VNX_SET, ptr, 4)){

    lsg[deviceID].Frequency = old_frequency;
    LockDev(deviceID, FALSE);// We're done using the frequency variables, unlock them..
    lsg[deviceID].pending = 0;
    return BAD_HID_IO;
  }
  LockDev(deviceID, FALSE);// We're done, let the status reports update frequency (in case of a sweep)
  lsg[deviceID].pending = 0;
  return STATUS_OK;
}

LVSTATUS fnLSG_SetStartFrequency(DEVID deviceID, int startfrequency) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;
  
  int old_startfrequency = lsg[deviceID].StartFrequency;

  if ((startfrequency >= lsg[deviceID].MinFrequency) && (startfrequency <= lsg[deviceID].MaxFrequency)) {
    lsg[deviceID].StartFrequency = startfrequency;
    if (TestMode)
      return STATUS_OK;// in test mode we update our internal variables, but don't talk to the real HW
  } else
    return BAD_PARAMETER;

  // the sweep start frequency value is OK, lets send it to the hardware
  unsigned char *ptr = (unsigned char *) &lsg[deviceID].StartFrequency;

  lsg[deviceID].pending = 1;
  usleep(10000); /* wait 10 ms for any reads to stop */
  if (DEBUG_OUT > 0) printf("deviceID = %d ptr = %x StartFrequency = %x\r\n", deviceID, ptr, *ptr);
  if (!SendReport(deviceID, VNX_FSTART | VNX_SET, ptr, 4)){
    lsg[deviceID].StartFrequency = old_startfrequency;
    lsg[deviceID].pending = 0;
    return BAD_HID_IO;
  }
  lsg[deviceID].pending = 0;
  return STATUS_OK;
}

LVSTATUS fnLSG_SetEndFrequency(DEVID deviceID, int endfrequency) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;
  
  int old_endfrequency = lsg[deviceID].EndFrequency;

  if ((endfrequency >= lsg[deviceID].MinFrequency) && (endfrequency <= lsg[deviceID].MaxFrequency)) {
    lsg[deviceID].EndFrequency = endfrequency;
    if (TestMode)
      return STATUS_OK;// in test mode we update our internal variables, but don't talk to the real HW
  } else
    return BAD_PARAMETER;

  // the sweep end frequency value is OK, lets send it to the hardware
  unsigned char *ptr = (unsigned char *) &lsg[deviceID].EndFrequency;

  if (DEBUG_OUT > 0) printf("deviceID = %d ptr = %x SweepEndFrequency = %x\r\n", deviceID, ptr, *ptr);
  
  lsg[deviceID].pending = 1;
  usleep(10000); /* wait 10 ms for any reads to stop */
  if (!SendReport(deviceID, VNX_FSTOP | VNX_SET, ptr, 4)){
    lsg[deviceID].EndFrequency = old_endfrequency;
    lsg[deviceID].pending = 0;
    return BAD_HID_IO;
  }
  lsg[deviceID].pending = 0;
  return STATUS_OK;
}

LVSTATUS fnLSG_SetDwellTime(DEVID deviceID, int dwelltime) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  int old_dwelltime = lsg[deviceID].DwellTime;

  if (dwelltime >= VNX_MIN_DWELLTIME) {
    lsg[deviceID].DwellTime = dwelltime;
    if (TestMode)
      return STATUS_OK;// in test mode we update our internal variables, but don't talk to the real HW
  } else
      return BAD_PARAMETER;
 
  // the dwell time value is OK, lets send it to the hardware
  unsigned char *ptr = (unsigned char *) &lsg[deviceID].DwellTime;

  if (DEBUG_OUT > 0) printf("deviceID = %d ptr = %x DwellTime = %x\r\n", deviceID, ptr, *ptr);
  
  lsg[deviceID].pending = 1;
  usleep(10000); /* wait 10 ms for any reads to stop */
  if (!SendReport(deviceID, VNX_FDWELL | VNX_SET, ptr, 4)){
    lsg[deviceID].DwellTime = old_dwelltime;
    lsg[deviceID].pending = 1;
    return BAD_HID_IO;
  }
  lsg[deviceID].pending = 1;
  return STATUS_OK;
}

LVSTATUS fnLSG_SetPowerLevel(DEVID deviceID, int powerlevel) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;
  
  int old_powerlevel = lsg[deviceID].Power;

  // We use an absolute power level for the API, so the attenuation is computed from the value set by the user
  
  // First off, check if the desired power setting is in range
  if ((powerlevel > lsg[deviceID].MaxPower) || (powerlevel < lsg[deviceID].MinPower))
    return BAD_PARAMETER;// power setting is out of range, bail out
  
  // We set the attenuation relative to MaxPower, so we have to subtract the power level value to get the attenuation
  // for powerlevel settings below 0 db we end up adding the value since it is negative.
  
  powerlevel = (lsg[deviceID].MaxPower - powerlevel);
  
  if ((powerlevel >= 0) && (powerlevel <= 4 * MAX_ATTEN)){
    lsg[deviceID].Power = powerlevel;
    if (TestMode)
      return STATUS_OK;// in test mode we update our internal variables, but don't talk to the real HW
  } else
    return BAD_PARAMETER;
  
  // the power level value is OK, lets send it to the hardware

  unsigned char *ptr = (unsigned char *) &lsg[deviceID].Power;

  if (DEBUG_OUT > 0) printf("deviceID = %d ptr = %x PowerLevel = %x\r\n", deviceID, ptr, *ptr);
  lsg[deviceID].pending = 1;
  usleep(10000); /* wait 10 ms for any reads to stop */
  if (!SendReport(deviceID, VNX_PWR | VNX_SET, ptr, 4)){
    lsg[deviceID].Power = old_powerlevel;
    lsg[deviceID].pending = 0;
    return BAD_HID_IO;
  }
  lsg[deviceID].pending = 0;
  return STATUS_OK;
}

LVSTATUS fnLSG_SetRFOn(DEVID deviceID, bool on) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
      return DEVICE_NOT_READY;

  char VNX_command[4] = {0, 0, 0, 0};

  if (on) {
    lsg[deviceID].Modebits = lsg[deviceID].Modebits & ~MODE_RFON;
    lsg[deviceID].Modebits = lsg[deviceID].Modebits | MODE_RFON;
    VNX_command[0] = 1;
  } else { // off
    lsg[deviceID].Modebits = lsg[deviceID].Modebits & ~MODE_RFON;
    VNX_command[0] = 0;
  }

  if (TestMode)
    return STATUS_OK; // in test mode we update our internal variables, but don't talk to the real HW

  lsg[deviceID].pending = 1;
  usleep(10000); /* wait 10 ms for any reads to stop */
  if (!SendReport(deviceID, VNX_RFMUTE | VNX_SET, VNX_command, 1)) {
    lsg[deviceID].pending = 1;
    return BAD_HID_IO;
  }

  lsg[deviceID].pending = 1;
  return STATUS_OK;
}

LVSTATUS fnLSG_SetFrequencyStep(DEVID deviceID, int frequencystep) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  int old_frequencystep = lsg[deviceID].FrequencyStep;

  if (frequencystep > 0 && frequencystep < (lsg[deviceID].MaxFrequency - lsg[deviceID].MinFrequency)){
    lsg[deviceID].FrequencyStep = frequencystep;
    if (TestMode)
      return STATUS_OK;// in test mode we update our internal variables, but don't talk to the real HW
  } else
    return BAD_PARAMETER;

  // the frequency step value is OK, lets send it to the hardware
  unsigned char *ptr = (unsigned char *) &lsg[deviceID].FrequencyStep;

  if (DEBUG_OUT > 0) printf("deviceID = %d ptr = %x Pulse On Time = %x\r\n", deviceID, ptr, *ptr);
  lsg[deviceID].pending = 1;
  usleep(10000); /* wait 10 ms for any reads to stop */

  if (!SendReport(deviceID, VNX_FSTEP | VNX_SET, ptr, 4)){
    lsg[deviceID].FrequencyStep = old_frequencystep;
  lsg[deviceID].pending = 0;
    return BAD_HID_IO;
  }
  lsg[deviceID].pending = 0;
  return STATUS_OK;
}

LVSTATUS fnLSG_SetUseInternalRef(DEVID deviceID, bool internal) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;
  
  char VNX_command[4] = {0, 0, 0, 0};

  if (internal) {
    lsg[deviceID].Modebits = lsg[deviceID].Modebits | MODE_INTREF;
    VNX_command[0] = 1;
  } else {
    lsg[deviceID].Modebits = lsg[deviceID].Modebits & ~MODE_INTREF;
    VNX_command[0] = 0;
  }

  if (TestMode)
    return STATUS_OK; // in test mode we update our internal variables, but don't talk to the real HW

  lsg[deviceID].pending = 1;
  usleep(10000); /* wait 10 ms for any reads to stop */
  if (!SendReport(deviceID, VNX_INTOSC | VNX_SET, VNX_command, 1)) {
    lsg[deviceID].pending = 0;
    return BAD_HID_IO;
  }

    lsg[deviceID].pending = 0;  
  return STATUS_OK;
}

LVSTATUS fnLSG_SetSweepDirection(DEVID deviceID, bool up) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;
  if (up)
    lsg[deviceID].Modebits = lsg[deviceID].Modebits & ~SWP_DIRECTION;// sweep direction up (bit == 0)
  else
    lsg[deviceID].Modebits = lsg[deviceID].Modebits | SWP_DIRECTION;// sweep direction downwards
  
  return STATUS_OK;
}

LVSTATUS fnLSG_SetSweepMode(DEVID deviceID, bool mode) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  if (mode) {
    lsg[deviceID].Modebits = lsg[deviceID].Modebits | SWP_CONTINUOUS;// Repeated sweep
    lsg[deviceID].Modebits = lsg[deviceID].Modebits & ~SWP_ONCE;
  } else {
    lsg[deviceID].Modebits = lsg[deviceID].Modebits | SWP_ONCE;// one time sweep
    lsg[deviceID].Modebits = lsg[deviceID].Modebits & ~SWP_CONTINUOUS;
  }

  return STATUS_OK;
}


LVSTATUS fnLSG_StartSweep(DEVID deviceID, bool go) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  char VNX_sweep[4] = {0, 0, 0, 0};

  if (go)
    VNX_sweep[0] = (char)lsg[deviceID].Modebits & MODE_SWEEP;
  else
    VNX_sweep[0] = 0;
  
  if (TestMode)
    return STATUS_OK;// in test mode we update our internal variables, but don't talk to the real HW

  if (DEBUG_OUT > 0) printf(" sending a sweep command = %x\r\n", VNX_sweep[0] );

  lsg[deviceID].pending = 1;
  usleep(10000); /* wait 10 ms for any reads to stop */
  if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_sweep, 1)) {
    lsg[deviceID].pending = 0;
    return BAD_HID_IO;
  }
  
  lsg[deviceID].pending = 0;
  return STATUS_OK;
}

LVSTATUS fnLSG_SaveSettings(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  if (TestMode)
    return STATUS_OK;// in test mode we update our internal variables, but don't talk to the real HW

  char VNX_savesettings[3] = {0x42, 0x55, 0x31}; //three byte key to unlock the user protection.

  lsg[deviceID].pending = 1;
  usleep(10000); /* wait 10 ms for any reads to stop */
  if (!SendReport(deviceID, VNX_SAVEPAR | VNX_SET, VNX_savesettings, 3)) {
    lsg[deviceID].pending = 0;
    return BAD_HID_IO;
  }
  
  lsg[deviceID].pending = 0;  
  return STATUS_OK;
}

int fnLSG_GetFrequency(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;
  
  return lsg[deviceID].Frequency;
}

int fnLSG_GetStartFrequency(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
      return DEVICE_NOT_READY;
  
  return lsg[deviceID].StartFrequency;
}

int fnLSG_GetEndFrequency(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;
  
  return lsg[deviceID].EndFrequency;
}

int fnLSG_GetDwellTime(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  return lsg[deviceID].DwellTime;
}

int fnLSG_GetFrequencyStep(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  return lsg[deviceID].FrequencyStep;
}

int fnLSG_GetRF_On(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
      return DEVICE_NOT_READY;

  if (lsg[deviceID].Modebits & MODE_RFON)
    return 1;
  else
    return 0;
}

int fnLSG_GetUseInternalRef(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  if (lsg[deviceID].Modebits & MODE_INTREF)
    return 1;
  else
    return 0;
}

int fnLSG_GetPowerLevel(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  return lsg[deviceID].Power;
}

int fnLSG_GetMaxPwr(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;
  
  return lsg[deviceID].MaxPower;
}

int fnLSG_GetMinPwr(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  if (!CheckDeviceOpen(deviceID))
    return DEVICE_NOT_READY;

  return lsg[deviceID].MinPower;}

int fnLSG_GetMaxFreq(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;

  return lsg[deviceID].MaxFrequency;
}

int fnLSG_GetMinFreq(DEVID deviceID) {
  if (deviceID >= MAXDEVICES)
    return INVALID_DEVID;
  
  return lsg[deviceID].MinFrequency;
}

